import { expect } from 'chai';
import { describe } from 'mocha';
import { deployContractsForTests, getSigners } from './helpers/deploy-helper';
import { Signers } from './types';
import { ethers } from 'hardhat';
import { PaymentToken } from '../../../types';
import { upgradeTransparentProxyContract } from '../../scripts/upgradeTransparentProxyContract';

describe('10 - Payment Token Contract Test', function () {
  let paymentToken: PaymentToken;
  let signers: Signers;
  let permissionsContract: any;

  beforeEach(async function () {
    const contracts = await deployContractsForTests();
    paymentToken = contracts.paymentTokenContract;
    signers = await getSigners();

    const MockPermissionFactory =
      await ethers.getContractFactory('MockPermission');
    permissionsContract = await MockPermissionFactory.deploy();
    await permissionsContract.deployed();

    await paymentToken.setPermissionContract(permissionsContract.address);

    // We will permit admin, signer1, and signer2. signer3 will be non-permitted.
    await permissionsContract.addAccounts([
      signers.admin.address,
      signers.signer1.address,
      signers.signer2.address,
    ]);
  });

  describe('ERC 20 standard tests:', function () {
    it('contract can be deployed', async function () {
      await expect(paymentToken).to.exist;
    });

    it('token should have name PaymentToken', async function () {
      await expect(await paymentToken.name()).is.equal('PaymentToken');
    });

    it('token should have FDE symbol', async function () {
      await expect(await paymentToken.symbol()).is.equal('FDE');
    });
    it('should mint payment tokens', async function () {
      const amountInWei = ethers.utils.parseEther('10');

      const data = {
        mintTo: signers.admin.address,
        mintAmount: amountInWei,
      };

      const jsonData: string = JSON.stringify(data);
      const dataAsBytes: string = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(jsonData),
      );

      await paymentToken.mint(data.mintTo, data.mintAmount, dataAsBytes);
      await expect(
        await paymentToken.balanceOf(signers.admin.address),
      ).is.equal(ethers.BigNumber.from(ethers.utils.parseEther('10')));
    });

    it('should upgrade contract without losing token data', async function () {
      // Mint some tokens
      const amountInWei = ethers.utils.parseEther('10');
      const data = {
        mintTo: signers.admin.address,
        mintAmount: amountInWei,
      };

      const jsonData: string = JSON.stringify(data);
      const dataAsBytes: string = ethers.utils.hexlify(
        ethers.utils.toUtf8Bytes(jsonData),
      );

      await paymentToken.mint(data.mintTo, data.mintAmount, dataAsBytes);
      await expect(
        await paymentToken.balanceOf(signers.admin.address),
      ).is.equal(ethers.BigNumber.from(ethers.utils.parseEther('10')));
      expect(await paymentToken.ver()).to.equal(1);

      // Upgrade contract
      await upgradeTransparentProxyContract({
        contractName: 'PaymentTokenV2',
        proxyContractAddress: paymentToken.address,
      });
      await expect(
        await paymentToken.balanceOf(signers.admin.address),
      ).is.equal(ethers.BigNumber.from(ethers.utils.parseEther('10')));
      expect(await paymentToken.ver()).to.equal(2);
    });
  });

  describe('contract extends OpenZeppelin Ownable ', function () {
    it('should check contract owner', async function () {
      expect(await paymentToken.owner()).to.equal(signers.admin.address);
    });
    it('owner can change the owner', async function () {
      await paymentToken.transferOwnership(signers.signer1.address);
      expect(await paymentToken.owner()).to.equal(signers.signer1.address);
    });
  });

  describe('Permission Logic (with MockPermissionContract)', function () {
    const amount = ethers.utils.parseEther('100');

    it('should allow a permitted account to transfer tokens', async function () {
      const permittedSigner = signers.signer1;
      const recipient = signers.signer3;

      await paymentToken.mint(permittedSigner.address, amount, '0x');

      const transferTx = paymentToken
        .connect(permittedSigner)
        .transfer(recipient.address, amount);

      await expect(transferTx).to.not.be.reverted;
    });

    it('should revert when a non-permitted account tries to transfer tokens', async function () {
      const nonPermittedSigner = signers.signer3;
      const recipient = signers.admin;

      await paymentToken.mint(nonPermittedSigner.address, amount, '0x');

      const transferTx = paymentToken
        .connect(nonPermittedSigner)
        .transfer(recipient.address, amount);

      await expect(transferTx).to.be.revertedWith(
        'PaymentToken: sender account is not permitted',
      );
    });

    it('should allow transfers after an account is added to the permission list', async function () {
      const initiallyNonPermitted = signers.signer3;
      const recipient = signers.admin;

      await paymentToken.mint(initiallyNonPermitted.address, amount, '0x');

      await expect(
        paymentToken
          .connect(initiallyNonPermitted)
          .transfer(recipient.address, amount),
      ).to.be.revertedWith('PaymentToken: sender account is not permitted');

      await permissionsContract.addAccounts([initiallyNonPermitted.address]);

      await expect(
        paymentToken
          .connect(initiallyNonPermitted)
          .transfer(recipient.address, amount),
      ).to.not.be.reverted;
    });

    it('should prevent transfers after a permitted account is removed', async function () {
      const initiallyPermitted = signers.signer2;
      const recipient = signers.signer3;

      if (!permissionsContract.removeAccount) {
        this.skip();
      }

      await paymentToken.mint(initiallyPermitted.address, amount, '0x');

      await expect(
        paymentToken
          .connect(initiallyPermitted)
          .transfer(recipient.address, amount.div(2)),
      ).to.not.be.reverted;

      await permissionsContract.removeAccount(initiallyPermitted.address);

      await expect(
        paymentToken
          .connect(initiallyPermitted)
          .transfer(recipient.address, amount.div(2)),
      ).to.be.revertedWith('PaymentToken: sender account is not permitted');
    });
  });
});
